/*
 * DrawDocMaze.java
 *
 * Created on March 4, 2003, 4:37 PM
 */

package nom.DannyBrewer.OOoUno;

import nom.DannyBrewer.utils.ExceptionHandler;

import nom.DannyBrewer.recreation.squareMaze.*;


//----------------------------------------------------------------------
//  UNO imports
//----------------------------------------------------------------------

// base of all interfaces
import com.sun.star.uno.XInterface;

// access the implementations via names
import com.sun.star.comp.servicemanager.ServiceManager;


import com.sun.star.lang.XMultiServiceFactory;
import com.sun.star.lang.XMultiComponentFactory;
import com.sun.star.connection.XConnector;
import com.sun.star.connection.XConnection;

import com.sun.star.bridge.XUnoUrlResolver;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.uno.XInterface;
import com.sun.star.uno.XNamingService;
import com.sun.star.uno.XComponentContext;


//----------------------------------------------------------------------
//  OpenOffice.org imports
//----------------------------------------------------------------------

// staroffice interfaces to provide desktop and componentloader
// and components i.e. spreadsheets, writerdocs etc.
import com.sun.star.frame.XDesktop;
import com.sun.star.frame.XComponentLoader;

// additional classes required
import com.sun.star.sheet.*;
import com.sun.star.container.*;
import com.sun.star.table.*;
import com.sun.star.beans.*;
import com.sun.star.style.*;
import com.sun.star.lang.*;
import com.sun.star.drawing.*;
import com.sun.star.text.XText;
import com.sun.star.awt.*;



/**
 * Instantiating this class creates a new Open Office Draw document.
 *
 * @author  danny brewer
 */
public class DrawDocMaze {	
	
    //----------------------------------------------------------------------
    //  Constructor
    //----------------------------------------------------------------------
	
	public DrawDocMaze()
				throws	java.lang.Exception,
						com.sun.star.uno.Exception,
						com.sun.star.connection.NoConnectException,
						com.sun.star.beans.UnknownPropertyException
	{
		Object oOORmtServiceMgr = OOUtils.getOORemoteServiceManager();
		initialize( oOORmtServiceMgr );
	}
	
	public DrawDocMaze( String host, String port )
				throws	java.lang.Exception,
						com.sun.star.uno.Exception,
						com.sun.star.connection.NoConnectException,
						com.sun.star.beans.UnknownPropertyException
	{
		Object oOORmtServiceMgr = OOUtils.getOORemoteServiceManager( host, port );
		initialize( oOORmtServiceMgr );
	}
	
	public DrawDocMaze( String unoRemoteServiceManagerUrl )
				throws	java.lang.Exception,
						com.sun.star.uno.Exception,
						com.sun.star.connection.NoConnectException,
						com.sun.star.beans.UnknownPropertyException
	{
		Object oOORmtServiceMgr = OOUtils.getOORemoteServiceManager( unoRemoteServiceManagerUrl );
		initialize( oOORmtServiceMgr );
	}
	
	public DrawDocMaze( Object oOORmtServiceMgr )
				throws	java.lang.Exception,
						com.sun.star.uno.Exception,
						com.sun.star.connection.NoConnectException,
						com.sun.star.beans.UnknownPropertyException
	{
		initialize( oOORmtServiceMgr );
	}
	

	//----------------------------------------------------------------------
    //  Public API
    //----------------------------------------------------------------------
	
	public void setMaze( SquareMaze maze ) {
		this.maze = maze;
		
		rows = maze.getRows();
		cols = maze.getCols();
	}
	public SquareMaze getMaze() { return maze; }
	
	
	public void drawMaze( SquareMaze maze ) {
		setMaze( maze );
		drawMaze();
	}
	
	public void drawMaze() {
		newDrawPage();
		drawHorizontalLines();
		drawVerticalLines();
	}
	
	
	//----------------------------------------------------------------------
    //  Internal Support -- drawing the maze
    //----------------------------------------------------------------------
	
	private SquareMaze maze = null;
	private int rows, cols;
	
	
	private void drawHorizontalLines() {
		for( int row = 0;  row <= rows;  ++row ) {
			int col = 0;
			while( col < cols ) {
				boolean showLine = showHLine( row, col );
				int width = hWidth( row, col, showLine );
				if( showLine ) drawCellHLine( row, col, width );
				col = col + width;
			}
		}
	}
	
	private boolean showHLine( int row, int col ) {
		boolean showLine = false;
		if( hasWall( row-1, col, SquareMaze.DIRECTION_BOTTOM ) ) {
			showLine = true;
		}
		else if( hasWall( row, col, SquareMaze.DIRECTION_TOP ) ) {
			showLine = true;
		}
		return showLine;
	}
	
	private int hWidth( int row, int col, boolean showLine ) {
		int width = 1;
		++col;
		while( col < cols ) {
			if( showLine != showHLine( row, col ) ) break;
			++width;
			++col;
		}
		return width;
	}
	
	
	private void drawVerticalLines() {
		for( int col = 0;  col <= cols;  ++col ) {
			int row = 0;
			while( row < rows ) {
				boolean showLine = showVLine( row, col );
				int height = vHeight( row, col, showLine );
				if( showLine ) drawCellVLine( row, col, height );
				row = row + height;
			}
		}
	}
	
	private boolean showVLine( int row, int col ) {
		boolean showLine = false;
		if( hasWall( row, col-1, SquareMaze.DIRECTION_RIGHT ) ) {
			showLine = true;
		}
		else if( hasWall( row, col, SquareMaze.DIRECTION_LEFT ) ) {
			showLine = true;
		}
		return showLine;
	}
	
	private int vHeight( int row, int col, boolean showLine ) {
		int width = 1;
		++row;
		while( row < rows ) {
			if( showLine != showVLine( row, col ) ) break;
			++width;
			++row;
		}
		return width;
	}
	
	
	//----------------------------------------------------------------------
    //  Internal Support -- maze access conveniences
    //----------------------------------------------------------------------
	
	private boolean hasWall( int row, int col, int direction ) {
		SquareMaze.MazeCell cell = maze.getMazeCell( row, col );
		if( cell != null ) {
			return cell.hasWall( direction );
		}
		return false;
	}
	
	private boolean isSolutionCell( int row, int col ) {
		SquareMaze.MazeCell cell = maze.getMazeCell( row, col );
		if( cell != null ) {
			return cell.isOnSolutionPath();
		}
		return false;
	}

	
	//----------------------------------------------------------------------
    //  Internal Support -- OpenOffice.org line drawing
    //----------------------------------------------------------------------
	
	private int mazeTop, mazeLeft, mazeBottom, mazeRight;
	private int mazeWidth, mazeHeight;
	private int cellWidth, cellHeight;

	protected void drawCellHLine( int row, int col, int width ) {
		int xStart = mazeLeft + (col * cellWidth);
		int xEnd = xStart + (width * cellWidth);
		int y = mazeTop + (row * cellHeight);
		drawLine( xStart, y, xEnd, y );
	}
	
	protected void drawCellVLine( int row, int col, int height ) {
		int x = mazeLeft + (col * cellWidth);
		int yStart = mazeTop + (row * cellHeight);
		int yEnd = yStart + (height * cellHeight);
		drawLine( x, yStart, x, yEnd );
	}
	
	protected void drawLine( int xStart, int yStart, int xEnd, int yEnd ) {
		int width = xEnd - xStart;
		int height = yEnd - yStart;
		XShape lineShape = OODrawUtils.createLineShape( drawDoc_XMultiServiceFactory, xStart, yStart, width, height );
		drawPage_XShapes.add( lineShape );
	}

	
	//----------------------------------------------------------------------
    //  Internal Support -- create drwing
    //----------------------------------------------------------------------
	
	// The drawing document.
	// Multiple interfaces to the same object.
	private XComponent				drawDoc_XComponent = null;
	private XDrawPagesSupplier		drawDoc_XDrawPagesSupplier = null;
	private XMultiServiceFactory	drawDoc_XMultiServiceFactory = null;
	
	
	private void initialize( Object oOORmtServiceMgr )
				throws com.sun.star.uno.Exception
	{
		// Create a new empty drawing document in the remote OpenOffice.org.
		drawDoc_XComponent = createDrawDocument( oOORmtServiceMgr );
		
		// Get some other useful interfaces to the same draw document object.
		drawDoc_XDrawPagesSupplier = QueryInterface.XDrawPagesSupplier( drawDoc_XComponent );
		drawDoc_XMultiServiceFactory = QueryInterface.XMultiServiceFactory( drawDoc_XComponent );
	}
	
	
	private XComponent createDrawDocument( Object oOORmtServiceMgr )
				throws com.sun.star.uno.Exception
	{
//		XMultiComponentFactory xMultiComponentFactory =
//				QueryInterface.XMultiComponentFactory( oOORmtServiceMgr );

		// Get the desired interface to the object.
		XMultiServiceFactory oOORmtServiceMgr_MultiServiceFactory =
				QueryInterface.XMultiServiceFactory( oOORmtServiceMgr );
		
		// Ask the MultiServiceFactory for a Desktop.
		// This might throw com.sun.star.uno.Exception.
		// This gives us the desktop, via. it's XInterface interface.
		XInterface desktop_XInterface =
				(XInterface) oOORmtServiceMgr_MultiServiceFactory.createInstance( "com.sun.star.frame.Desktop" );
		
		// Get the desired interface from the object.
		// We don't need the XInterface interface, we need the XComponentLoader interface to the object.
		XComponentLoader desktop_XComponentLoader =
				QueryInterface.XComponentLoader( desktop_XInterface );
		
		PropertyValue [] szEmptyArgs = new PropertyValue [0];
		String url = "private:factory/sdraw";
		// Ask the Desktop's component loader to load the url for a new Draw document.
		// This might throw com.sun.star.uno.Exception.
		// This gives us a drawing document.
		// It gives us the XComponent interface to the drawing.
		// There are other interfaces to this object available via. QueryInterface.
		XComponent drawDoc_XComponent = desktop_XComponentLoader.loadComponentFromURL( url, "_blank", 0, szEmptyArgs );
		
		return drawDoc_XComponent;
	}
	
	
	
	// The index of the page we're actively drawing on.
	int curPageNum = -1;
	
	// The current page of the drawing.
	// Multiple interfaces to the same object.
	private XDrawPage				drawPage_XDrawPage = null;
	private XShapes					drawPage_XShapes = null;
	
	// Page information
	int pageHeight, pageWidth;
	int borderTop, borderLeft, borderRight, borderBottom;
	

	// Create a new drawing page.
	private void newDrawPage() {
		curPageNum = curPageNum + 1;
		
		int numDrawPages = OODrawUtils.getNumDrawPages( drawDoc_XDrawPagesSupplier );
		if( curPageNum < numDrawPages ) {
			// Get page of the draw document.
			drawPage_XDrawPage = OODrawUtils.getXDrawPage( drawDoc_XDrawPagesSupplier, curPageNum );
		} else {
			// Create a new page.
			drawPage_XDrawPage = OODrawUtils.insertNewPageByIndex( drawDoc_XDrawPagesSupplier, curPageNum );
		}
		
		// Get some other useful interfaces to the same draw document object.
		drawPage_XShapes = QueryInterface.XShapes( drawPage_XDrawPage );
		
		// Set the page name.
		OODrawUtils.setPageName( drawPage_XDrawPage, "Maze " + (curPageNum + 1) );
		
		// Get dimensions of the page.
		pageHeight = OOUtils.getIntProperty( drawPage_XDrawPage, "Height" );
		pageWidth  = OOUtils.getIntProperty( drawPage_XDrawPage, "Width" );
		
		// Get margins
		borderTop = OOUtils.getIntProperty( drawPage_XDrawPage, "BorderTop" );
		borderLeft = OOUtils.getIntProperty( drawPage_XDrawPage, "BorderLeft" );
		borderRight = OOUtils.getIntProperty( drawPage_XDrawPage, "BorderRight" );
		borderBottom = OOUtils.getIntProperty( drawPage_XDrawPage, "BorderBottom" );
		
		// centimeters * 1000, so 500 means 0.5 cm
		mazeTop = borderTop + 500;
		mazeLeft = borderLeft + 500;
		mazeBottom = borderRight + 500;
		mazeRight = borderBottom + 500;
		
		mazeWidth = pageWidth - mazeLeft - mazeRight;
		mazeHeight = pageHeight - mazeTop - mazeBottom;
		
		cellWidth = mazeWidth / cols;
		cellHeight = mazeHeight / rows;
	}
	
		
	//----------------------------------------------------------------------
    //  Test Program
    //----------------------------------------------------------------------
	
    public static void main( String[] args ) {
		DrawDocMaze drawDocMaze = null;
		try {
			drawDocMaze = new DrawDocMaze();
        }
        catch ( java.lang.Exception e ) {
			e.printStackTrace();
			System.exit( 0 );
        }
		
		SquareMaze maze = new SquareMaze( 20, 15 );
		SquareMazeGenerator1 mazeMaker = new SquareMazeGenerator1( maze );
		mazeMaker.generateMaze();
		
		// Draw the current maze.
		drawDocMaze.drawMaze( maze );

		// Draw a couple more new mazes.
		maze.initializeMazeCells();
		mazeMaker.generateMaze();
		drawDocMaze.drawMaze( maze );

		maze.initializeMazeCells();
		mazeMaker.generateMaze();
		drawDocMaze.drawMaze( maze );
		
		System.exit( 0 );
    }
	
	
}

